//-----------------------------------------------------------------------------
// Remote Boss Example for C#Builder
//
// RemoteBoss.cs
//
// This is the common assembly used by the client and server applications for
// the "Remote Boss" Remoting example. It demonstrates:
// 1) Capture the screen image in PNG format and packaging it for remoting.
// 2) Using a thread-safe list of strings.
// 3) Importing Win32 API unmanaged DLL functions.
// 4) Starting an external process with command line arguments.
//
// By Jarrod Hollingworth
//
//-----------------------------------------------------------------------------
using System;
using System.Runtime.Serialization;
// ArrayList
using System.Collections;
// Start a process
using System.Diagnostics;
// Screen capture.
using System.Windows.Forms;
using System.Drawing;
using System.Drawing.Imaging;
using System.Runtime.InteropServices;
using System.IO;

namespace RemoteBoss
{
  // Thread-safe access to new messages from the boss. This is implemented
  // as a singleton (only one instance of TMessageList exists).
  public class BossMessageList
  {
    private static BossMessageList Instance = new BossMessageList();
    public ArrayList MessageList;

    private BossMessageList()
    {
      MessageList = new ArrayList();
    }

    public static BossMessageList GetInstance()
    {
      return Instance;
    }

    // Add a message to the list.
    public void AddMessage(string Message)
    {
      lock(MessageList.SyncRoot);
      MessageList.Add("=============================================\r\n" +
                      "Message From the Boss at " + DateTime.Now.ToString() + "\r\n" +
                      Message);
    }

    // Get all messages in the list for display.
    // NOTE: The messages returned are removed from the list.
    public string GetMessages()
    {
      // Get and remove messages.
      string Messages = "";
      lock(MessageList.SyncRoot);
      foreach (string msg in MessageList) {
        Messages += msg + "\r\n";
      }
      MessageList.Clear();

      return Messages;
    }
  }

  /// <summary>
  /// Wrapper for imported Win32 API unmanaged functions.
  /// </summary>
  class Win32
  {
    [DllImport("user32.dll")]
    public static extern IntPtr GetDesktopWindow();

    [DllImport("user32.dll")]
    public static extern IntPtr GetWindowDC(IntPtr hWnd);

    [DllImport("user32.dll")]
    public static extern Int32 ReleaseDC(
      IntPtr hWnd,
      IntPtr hDC
      );

    [DllImport("gdi32.dll")]
    public static extern bool BitBlt(
      IntPtr hdcDest, // handle to destination device context
      int nXDest,     // x-coordinate of destination rectangle's upper-left corner
      int nYDest,     // y-coordinate of destination rectangle's upper-left corner
      int nWidth,     // width of destination rectangle
      int nHeight,    // height of destination rectangle
      IntPtr hdcSrc,  // handle to source device context
      int nXSrc,      // x-coordinate of source rectangle's upper-left corner
      int nYSrc,      // y-coordinate of source rectangle's upper-left corner
      Int32 dwRop     // raster operation code
      );
  }

  /// <summary>
  /// A screen capture of the PC so that the boss can spy on the workers.
  /// </summay>
  /// <remarks>
  /// This class is Serializable because objects will be sent from one
  /// AppDomain to another in different processes. The image is sent as
  /// a stream because Image cannot be serialized.
  /// </remarks>
  [Serializable]
  public class BossScreen
  {
    private DateTime CaptureDateTime;
    private MemoryStream ImageStream;

    private const Int32 SRCCOPY = 0xCC0020;

    public BossScreen()
    {
      CaptureDateTime = DateTime.Now;

      // Capture the screen.
      try {
        // Create a bitmap the same size as the screen.
        Image ScreenCapture = new Bitmap(Screen.PrimaryScreen.Bounds.Width,
                                   Screen.PrimaryScreen.Bounds.Height);
        // Get the device context for our bitmap.
        Graphics gr1 = Graphics.FromImage(ScreenCapture);
        IntPtr dc1 = gr1.GetHdc();
        try {
          // Get the device context of the desktop.
          IntPtr dw = Win32.GetDesktopWindow();
          IntPtr dc2 = Win32.GetWindowDC(dw);
          try {
            // Copy the desktop to our bitmap.
            Win32.BitBlt(dc1, 0, 0,
                         Screen.PrimaryScreen.Bounds.Width,
                         Screen.PrimaryScreen.Bounds.Height,
                         dc2, 0, 0, SRCCOPY);
          }
          finally {
            // Cleanup.
            Win32.ReleaseDC(dw, dc2);
          }
        }
        finally {
          // Cleanup.
          gr1.ReleaseHdc(dc1);
        }

        // Convert the image to a stream for sending to the client.
        ImageStream = new MemoryStream();
        ScreenCapture.Save(ImageStream, ImageFormat.Png);
      }
      catch (Exception ex) {
        MessageBox.Show("Exception: " + ex.Message);
      }
    }

    public DateTime GetCaptureDateTime() { return CaptureDateTime; }
    public MemoryStream GetImageStream() { return ImageStream; }
}

 	/// <summary>
	/// A message to send to a remote boss server including a process or file
  /// to run.
	/// </summary>
  /// <remarks>
  /// This class is Serializable because objects will be sent from one
  /// AppDomain to another in different processes.
  /// </remarks>
  [Serializable]
  public class BossMessage
  {
    private string Message;

    public BossMessage(string iMessage)
    {
      Message = iMessage;
    }

    public string GetMessage() { return Message; }
  }

  /// <summary>
	/// Details of a process to run on the workers PC and its command line
  /// arguments.
	/// </summary>
  /// <remarks>
  /// This class is Serializable because objects will be sent from one
  /// AppDomain to another in different processes.
  /// </remarks>
  [Serializable]
  public class BossProcess
  {
    private string FileName;
    private string CommandLineArgs;

    public BossProcess(string iFileName,
                       string iCommandLineArgs)
    {
      FileName = iFileName;
      CommandLineArgs = iCommandLineArgs;
    }

    public string GetFileName() { return FileName; }
    public string GetCommandLineArgs() { return CommandLineArgs; }
  }

 	/// <summary>
  /// The remote boss service.
	/// </summary>
  public class RemoteBossService : MarshalByRefObject
   {
    public BossScreen GetScreen()
    {
      return new BossScreen();
    }

    public void SendMessage(BossMessage Message)
    {
      // Add the message to be displayed. This is picked up by the UI.
      BossMessageList.GetInstance().AddMessage(Message.GetMessage());
    }

    public void PushProcess(BossProcess PushProcess)
    {
      // Start any associated process / file.
      if (PushProcess.GetFileName().Length > 0) {
        Process ProcessToStart = new Process();
        ProcessToStart.StartInfo.Arguments = PushProcess.GetCommandLineArgs();
        ProcessToStart.StartInfo.FileName = PushProcess.GetFileName();
        ProcessToStart.Start();
      }
    }
  }
}
